'use client';

import React, { useState } from 'react';
import { SCRIPT_TEMPLATES } from '@/lib/constants';
import { GlassCard } from '../components/ui/GlassCard';
import { Button } from '../components/ui/Button';
import { Search, PenTool, Palette, BarChart, MapPin, Container, BookOpen, FolderTree, Star } from 'lucide-react';

const IconMap: Record<string, React.ElementType> = {
  PenTool, Palette, BarChart, Search, MapPin, Container, BookOpen, FolderTree
};

export default function LibraryPage() {
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedCategory, setSelectedCategory] = useState<string>('All');

  const categories = ['All', 'AI', 'SEO', 'Data', 'Utilities', 'Maps'];

  const filteredScripts = SCRIPT_TEMPLATES.filter(script => {
    const matchesSearch = script.title.toLowerCase().includes(searchTerm.toLowerCase()) || 
                          script.description.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesCategory = selectedCategory === 'All' || script.category === selectedCategory;
    return matchesSearch && matchesCategory;
  });

  return (
    <div className="container mx-auto px-4 pt-24 pb-12">
      <div className="mb-8">
        <h1 className="text-3xl font-bold text-slate-900 mb-4">کتابخانه اسکریپت‌ها</h1>
        
        <div className="flex flex-col md:flex-row gap-4 justify-between items-center">
          {/* Search */}
          <div className="relative w-full md:w-96">
            <Search className="absolute right-3 top-1/2 -translate-y-1/2 text-slate-400 w-5 h-5" />
            <input 
              type="text"
              placeholder="جستجو در اسکریپت‌ها..."
              className="w-full bg-white border border-slate-200 rounded-xl py-3 pr-10 pl-4 text-slate-900 placeholder-slate-400 focus:outline-none focus:border-purple-500 focus:ring-1 focus:ring-purple-500 transition-all shadow-sm"
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
            />
          </div>

          {/* Categories */}
          <div className="flex gap-2 overflow-x-auto pb-2 w-full md:w-auto no-scrollbar">
            {categories.map(cat => (
              <button
                key={cat}
                onClick={() => setSelectedCategory(cat)}
                className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors whitespace-nowrap ${
                  selectedCategory === cat 
                    ? 'bg-purple-600 text-white' 
                    : 'bg-white border border-slate-200 text-slate-600 hover:bg-slate-50'
                }`}
              >
                {cat === 'All' ? 'همه' : cat}
              </button>
            ))}
          </div>
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
        {filteredScripts.map(script => {
          const Icon = IconMap[script.iconName] || PenTool;
          return (
            <GlassCard key={script.id} className="p-6 flex flex-col h-full hover:border-purple-500/40 transition-colors group">
              <div className="flex justify-between items-start mb-4">
                <div className="w-12 h-12 rounded-xl bg-slate-100 flex items-center justify-center group-hover:bg-purple-100 transition-colors">
                  <Icon className="w-6 h-6 text-slate-500 group-hover:text-purple-600" />
                </div>
                {script.popular && (
                  <span className="bg-amber-50 text-amber-600 text-xs px-2 py-1 rounded-full flex items-center gap-1 border border-amber-200">
                    <Star className="w-3 h-3 fill-current" />
                    محبوب
                  </span>
                )}
              </div>
              
              <h3 className="text-lg font-bold text-slate-900 mb-2">{script.title}</h3>
              <p className="text-slate-600 text-sm mb-6 flex-grow leading-relaxed">
                {script.description}
              </p>

              <div className="flex gap-2 mt-auto">
                <Button size="sm" className="w-full">نصب</Button>
                <Button size="sm" variant="outline" className="w-full">جزئیات</Button>
              </div>
            </GlassCard>
          );
        })}
      </div>
      
      {filteredScripts.length === 0 && (
        <div className="text-center py-20 text-slate-500">
          اسکریپتی با این مشخصات پیدا نشد.
        </div>
      )}
    </div>
  );
}

